

%% Step 1: Merge EddyPro output with input data (Loop through all folders)-SRSP emissions
clear all;
close all;
clc;

% Base directories
baseInputDir = '\\csunas01.colostate.edu\research\metec\user\mbua\ADED\Eddy Covariance\8_Hz\Event Tables\Emission Sources\Single_Point\';
baseOutputDir = 'C:\Users\mbua\Desktop\ADED\Eddy Covariance\Single_Point\';

% Define time intervals and angles to loop through
timeIntervals = {'30 minutes'};
angles = { '45 degrees'};

% Loop through each time interval
for t = 1:length(timeIntervals)
    timeDir = timeIntervals{t};
    
    % Loop through each angle
    for a = 1:length(angles)
        angleDir = angles{a};
        
        % Define input and output paths for current iteration
        inputDir = fullfile(baseInputDir, timeDir, angleDir);
        outputDirPath = fullfile(baseOutputDir, timeDir, angleDir, 'EddyPro Input\');
        
        fprintf('Processing: %s - %s\n', timeDir, angleDir);
        
        % Find all '*full_output*.csv' files in outputDirPath
        outputFiles = dir(fullfile(outputDirPath, '*full_output*.csv'));
        
        if isempty(outputFiles)
            warning('No matching output files found in: %s', outputDirPath);
            continue; % Skip to next angle if no output file exists
        else
            % Use the first matching file (modify if multiple files are expected)
            outputFilePath = fullfile(outputFiles(1).folder, outputFiles(1).name);
            fprintf('Using output file: %s\n', outputFilePath);
        end
        
        % Create output directory for merged files
        outputDir = fullfile(inputDir, 'EC Output');
        if ~exist(outputDir, 'dir')
            mkdir(outputDir);
        end
        
        % Read the EddyPro output file
        outputTable = readtable(outputFilePath);
        
        % Extract and rename relevant columns
        eddyTable = table();
        eddyTable.filename = outputTable.file_info;
        eddyTable.CH4_flux = outputTable.Var12;
        eddyTable.turbulence = outputTable.turbulence;
        eddyTable.L = outputTable.Var50;
        eddyTable.xPeak = outputTable.Var55;
        
        % Get input files
        inputFiles = dir(fullfile(inputDir, '*.csv'));
        
        % Process each input file
        for i = 1:length(inputFiles)
            inputFile = fullfile(inputFiles(i).folder, inputFiles(i).name);
            inputData = readtable(inputFile);
            
            % Extract start time from input filename (before " to ")
            inputName = erase(inputFiles(i).name, '.csv');
            tokens = split(inputName, ' to ');
            if numel(tokens) < 1
                warning('Invalid filename format: %s', inputFiles(i).name);
                continue;
            end
            startTimeStr = strtrim(tokens{1}); % e.g. '2024-02-13 10-00-11'
            try
                dt = datetime(startTimeStr, 'InputFormat', 'yyyy-MM-dd HH-mm-ss');
            catch
                warning('Invalid datetime format in: %s', startTimeStr);
                continue;
            end
            matchKey = sprintf('%s %02d%02d', datestr(dt, 'yyyy-mm-dd'), hour(dt), minute(dt)); % e.g. '2024-02-13 1000'
            
            % Match with eddyTable
            matchFound = false;
            for j = 1:height(eddyTable)
                eddyRaw = eddyTable.filename{j}; % e.g. '2024-02-13_1000_10-30-10.txt'
                eddyParts = split(eddyRaw, '_');
                if numel(eddyParts) < 2
                    continue;
                end
                eddyDate = eddyParts{1};         % '2024-02-13'
                eddyTime = eddyParts{2};         % '1000'
                eddyKey = [eddyDate ' ' eddyTime];
                
                if strcmp(eddyKey, matchKey)
                    % Append eddy data to inputData
                    inputData.CH4_flux = repmat(eddyTable.CH4_flux(j), height(inputData), 1);
                    inputData.turbulence = repmat(eddyTable.turbulence(j), height(inputData), 1);
                    inputData.L = repmat(eddyTable.L(j), height(inputData), 1);
                    inputData.xPeak = repmat(eddyTable.xPeak(j), height(inputData), 1);
                    
                    % Save to output directory
                    outputFile = fullfile(outputDir, inputFiles(i).name);
                    writetable(inputData, outputFile);
                    matchFound = true;
                    break;
                end
            end
            
            if ~matchFound
                warning('No match found for input file: %s', inputFiles(i).name);
            end
        end
    end
end

disp('All folders processed successfully!');




%% Step 2: Merge EddyPro output with input data (Loop through all folders)-MRSP emissions
clear all;
close all;
clc;

% Base directories
baseInputDir = '\\csunas01.colostate.edu\research\metec\user\mbua\ADED\Eddy Covariance\8_Hz\Event Tables\Emission Sources\Multi_Point\Emission Point\';
baseOutputDir = 'C:\Users\mbua\Desktop\ADED\Eddy Covariance\Multi_Point\';

% Define time intervals and angles to loop through
timeIntervals = {'5 minutes','15 minutes','30 minutes'};
angles = { '10 degrees', '20 degrees','45 degrees'};

% Loop through each time interval
for t = 1:length(timeIntervals)
    timeDir = timeIntervals{t};
    
    % Loop through each angle
    for a = 1:length(angles)
        angleDir = angles{a};
        
        % Define input and output paths for current iteration
        inputDir = fullfile(baseInputDir, timeDir, angleDir);
        outputDirPath = fullfile(baseOutputDir, timeDir, angleDir, 'EddyPro Input\');
        
        fprintf('Processing: %s - %s\n', timeDir, angleDir);
        
        % Find all '*full_output*.csv' files in outputDirPath
        outputFiles = dir(fullfile(outputDirPath, '*full_output*.csv'));
        
        if isempty(outputFiles)
            warning('No matching output files found in: %s', outputDirPath);
            continue; % Skip to next angle if no output file exists
        else
            % Use the first matching file (modify if multiple files are expected)
            outputFilePath = fullfile(outputFiles(1).folder, outputFiles(1).name);
            fprintf('Using output file: %s\n', outputFilePath);
        end
        
        % Create output directory for merged files
        outputDir = fullfile(inputDir, 'EC Output');
        if ~exist(outputDir, 'dir')
            mkdir(outputDir);
        end
        
        % Read the EddyPro output file
        outputTable = readtable(outputFilePath);
        
        % Extract and rename relevant columns
        eddyTable = table();
        eddyTable.filename = outputTable.file_info;
        eddyTable.CH4_flux = outputTable.Var12;
        eddyTable.turbulence = outputTable.turbulence;
        eddyTable.L = outputTable.Var50;
        eddyTable.xPeak = outputTable.Var55;
        
        % Get input files
        inputFiles = dir(fullfile(inputDir, '*.csv'));
        
        % Process each input file
        for i = 1:length(inputFiles)
            inputFile = fullfile(inputFiles(i).folder, inputFiles(i).name);
            inputData = readtable(inputFile);
            
    % Extract start time from input filename (before " to ")
inputName = erase(inputFiles(i).name, '.csv');
tokens = split(inputName, ' to ');
if numel(tokens) < 1
    warning('Invalid filename format: %s', inputFiles(i).name);
    continue;
end
% Extract only the first datetime part (before '_to_')
if contains(inputName, '_to_')
    datetimePart = extractBefore(inputName, '_to_');
else
    warning('Filename does not contain "_to_": %s', inputName);
    continue;
end
startTimeStr = strtrim(datetimePart); % '2024-02-11 09-30-37'

try
    % Parse the input file's datetime
    dt = datetime(startTimeStr, 'InputFormat', 'yyyy-MM-dd HH-mm-ss');
    
    % Create match key in format similar to EddyPro (date_HHMM)
    matchKey = sprintf('%s_%02d%02d', datestr(dt, 'yyyy-mm-dd'), hour(dt), minute(dt));
    
    % Match with eddyTable
    matchFound = false;
    for j = 1:height(eddyTable)
        eddyRaw = eddyTable.filename{j}; % e.g. '2024-02-08_2030_5S34.txt'
        
        % Extract just the date_HHMM part (before first underscore after time)
        eddyParts = split(eddyRaw, '_');
        if numel(eddyParts) < 2
            continue;
        end
        eddyKey = [eddyParts{1} '_' eddyParts{2}]; % '2024-02-08_2030'
        
        if strcmp(eddyKey, matchKey)
            % Append eddy data to inputData
            inputData.CH4_flux = repmat(eddyTable.CH4_flux(j), height(inputData), 1);
            inputData.turbulence = repmat(eddyTable.turbulence(j), height(inputData), 1);
            inputData.L = repmat(eddyTable.L(j), height(inputData), 1);
            inputData.xPeak = repmat(eddyTable.xPeak(j), height(inputData), 1);
            
            % Save to output directory
            outputFile = fullfile(outputDir, inputFiles(i).name);
            writetable(inputData, outputFile);
            matchFound = true;
            break;
        end
    end

catch ME
    warning('Error processing datetime in: %s. Error: %s', startTimeStr, ME.message);
    continue;
end
        end
    end
end

disp('All folders processed successfully!');

%% Step 3: Calculate footprint - Kljun
clear all;
close all;
clc;

% Define parent directory and output directory
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables\Emission Sources\Multi_Point\Emission Point';

% Time intervals and degree folders to process
minute_folders = {'5 minutes','15 minutes','30 minutes'};
degree_folders = {'10 degrees','20 degrees','45 degrees'};

% Constants for footprint calculation
zm = 2.9333;      % Measurement height above displacement height [m]
z0 = 0.1;    % Roughness length [m]

for m = 1:numel(minute_folders)
    folderName = minute_folders{m};
    minutePath = fullfile(parentPath, folderName);
    
    % Process only specified degree folders
    for d = 1:numel(degree_folders)
        degreeFolder = degree_folders{d};
        degreePath = fullfile(minutePath, degreeFolder, 'EC output');
        
        % Skip if folder doesn't exist
        if ~exist(degreePath, 'dir')
            warning('Folder not found: %s', degreePath);
            continue;
        end
        
        % Create output directory for footprint results
        outputPath = fullfile(degreePath, 'Footprint_Klujn');
        if ~exist(outputPath, 'dir')
            mkdir(outputPath);
        end
        
        % Get all CSV files in the directory
        files = dir(fullfile(degreePath, '*.csv'));
        
        for k = 1:numel(files)
            filename = fullfile(degreePath, files(k).name);
            try
                data = readtable(filename);
                
                % ===== Part 1: Determine boundary layer height (h) =====
                L_values = data.L;
                h_values = zeros(size(L_values));
                
                % Classify boundary layer height based on L values
                h_values(L_values <= 0 & L_values > -5000) = 2000;  % Unstable conditions
                h_values(L_values <= -5000 | L_values > 5000) = 1000; % Very unstable/stable
                h_values(L_values > 0 & L_values < 5000) = 800;     % Stable conditions
                
                % Add h_values to the table
                data.h = h_values;
                
                % ===== Calculate target coordinates =====
                target_x = mean(data.Distance_meters .* cosd(data.diff));
                target_y = mean(data.Distance_meters .* sind(data.diff));
                
                % ===== Footprint calculation parameters =====
                umean = mean(data.WS);          % Mean wind speed
                h = mean(data.h);               % Boundary layer height
                ol = mean(data.L);              % Obukhov length
                sigmav = std(data.V);           % Std of lateral velocity
                ustar = mean(data.turbulence);  % Friction velocity
                wind_dir = mean(data.WD);       % Wind direction
                
                % ===== Calculate footprint =====
                [FFP, flag_err] = calc_footprint_FFP(zm, z0, umean, h, ol, ustar, sigmav, wind_dir, 'r', 90);
                
                if ~flag_err
                    % Check if target is within footprint
                    in_area = inpolygon(target_x, target_y, FFP.xr(~isnan(FFP.xr)), FFP.yr(~isnan(FFP.yr)));
                    
                    if in_area
                        % Find closest grid point to target
                        [~, idx] = min((FFP.x_2d - target_x).^2 + (FFP.y_2d - target_y).^2, [], 'all', 'linear');
                        [idx_row, idx_col] = ind2sub(size(FFP.f_2d), idx);
                        f_2d_target = FFP.f_2d(idx_row, idx_col);
                        
                        % Store results
                        data.footprint_Klujn = repmat(f_2d_target, height(data), 1);
                        
                        % Save updated table
                        outputFile = fullfile(outputPath, files(k).name);
                        writetable(data, outputFile);
                        fprintf('Successfully processed: %s\n', files(k).name);
                    else
                        warning('Target (%.1f, %.1f) outside footprint area in: %s', target_x, target_y, files(k).name);
                    end
                else
                    warning('Footprint calculation failed for: %s', files(k).name);
                end
                
            catch ME
                warning('Error processing file %s: %s', files(k).name, ME.message);
            end
        end
    end
end

disp('Step 2 Complete - Footprint calculation finished');

%% Step 4: Calculate K&M footprint

clear all; close all; clc;

% Define parent directory and output directory
parentPath = '\\csunas01.colostate.edu\research\METEC\User\mbua\ADED\Eddy Covariance\8_Hz\Event Tables\Emission Sources\Multi_Point\Emission Point';
minute_folders = {'5 minutes','15 minutes','30 minutes'};
degree_folders = {'10 degrees','20 degrees','45 degrees'};

for m = 1:numel(minute_folders)
    folderName = minute_folders{m};
    minutePath = fullfile(parentPath, folderName);
    
    % Process only specified degree folders
    for d = 1:numel(degree_folders)
        degreeFolder = degree_folders{d};
        degreePath = fullfile(minutePath, degreeFolder, 'EC output');
        
        % Skip if folder doesn't exist
        if ~exist(degreePath, 'dir')
            warning('Folder not found: %s', degreePath);
            continue;
        end
        
        % Create output directory for footprint results
        outputPath = fullfile(degreePath, 'KM_Footprint');
        if ~exist(outputPath, 'dir')
            mkdir(outputPath);
        end
        
        files = dir(fullfile(degreePath,'*.csv'));
        
        for k = 1:numel(files)
 try
    filename = fullfile(degreePath, files(k).name);
    data = readtable(filename);
    
    % Constants
    z = 2.933;  % measurement height (m)
    kappa = 0.4; % von Karman constant
    z0 = 0.1;   % roughness length (m)
    
    % Calculate stability parameters (keep your original equations)
    phi_m = nan(height(data), 1);
    n = nan(height(data), 1);
    
    stable_idx = data.L > 0;
    phi_m(stable_idx) = 1 + 5 * z ./ data.L(stable_idx);
    n(stable_idx) = 1 ./ (1 + 5 * z ./ data.L(stable_idx));
    
    unstable_idx = data.L < 0;
    phi_m(unstable_idx) = (1 - 16 * z ./ data.L(unstable_idx)).^(-0.25);
    n(unstable_idx) = (1 - 24 * z ./ data.L(unstable_idx)) ./ (1 - 16 * z ./ data.L(unstable_idx));
    
    neutral_idx = data.L == 0;
    phi_m(neutral_idx) = 1;
    n(neutral_idx) = 1;
    
    m = data.turbulence .* phi_m ./ (kappa * mean(data.WS));
    r = 2 + m - n;
    mu = (1 + m) ./ r;
    data.flux_length = data.xPeak./(1+mu);
    
    % Use MEAN values for grid calculations
    mean_mu = mean(mu, 'omitnan');
    
    mean_ustar = mean(data.turbulence, 'omitnan');
    sigma_v = std(data.V, 'omitnan');
% Revised flux length calculation (Kormann & Meixner 2001 Eq. 16)
%mean_flux_length = (z^mean_mu) / (mean_mu * kappa * mean_ustar) .* ...
                 % (gamma(mean_mu)/gamma(mean_mu + 1));
                 mean_flux_length = mean(data.flux_length);
    % Diagnostic output
fprintf(['\n--- Parameter Diagnostics ---\n'...
         'Measurement height (z): %.3f m\n'...
         'Stability length (L): %.3f m\n'...
         'Wind speed (WS): %.3f m/s\n'...
         'u*: %.3f m/s\n'...
         'sigma_v: %.3f m/s\n'...
         'sigma_v/u* ratio: %.3f\n'...
         'Calculated flux length: %.3f m\n'...
         'Calculated mu: %.3f\n'],...
         z, mean(data.L), mean(data.WS), mean_ustar, sigma_v,...
         sigma_v/mean_ustar, mean_flux_length, mean_mu);

    % Grid settings
    x_vals = linspace(1, 300, 200); % along-wind distance (m)
    y_vals = linspace(-100, 100, 200); % cross-wind distance (m)
    [X, Y] = meshgrid(x_vals, y_vals);
    
    % 1. Calculate along-wind component (f_x)
    f_x = (1/gamma(mean_mu)) * (mean_flux_length^mean_mu ./ X.^(1+mean_mu)) .* exp(-mean_flux_length./X);
    
    % 2. Calculate crosswind spread (σy)
    sigma_y = 0.3 * z0 * (sigma_v/mean_ustar) .* (X/z0).^0.86;
    
    % 3. Calculate 2D footprint
    F = f_x .* (1./(sqrt(2*pi).*sigma_y)) .* exp(-Y.^2./(2.*sigma_y.^2));
    F_normalized = F ./ sum(F(:), 'omitnan'); % Normalize
                
                % Find 90% contour
                sorted_F = sort(F_normalized(:), 'descend');
                cumsum_F = cumsum(sorted_F);
                threshold_val = sorted_F(find(cumsum_F >= 0.9, 1));
                
                % Create contour
                C = contourc(x_vals, y_vals, F_normalized, [threshold_val threshold_val]);
                
                % Extract contour coordinates
                contour_x = [];
                contour_y = [];
                if ~isempty(C)
                    idx = 1;
                    while idx < size(C,2)
                        level = C(1,idx);
                        n_points = C(2,idx);
                        contour_x = [contour_x; C(1,idx+1:idx+n_points)'];
                        contour_y = [contour_y; C(2,idx+1:idx+n_points)'];
                        idx = idx + n_points + 1;
                    end
                end
                

            data.downwind = mod(data.bearing - 180, 360);
            data.diff = abs(data.downwind - data.avg_WD);
            data.diff = min(data.diff, 360 - data.diff); % Handle angular wrap-around

                % Calculate target point coordinates
                target_x = mean(data.Distance_meters.*cosd(data.diff));
                target_y = mean(data.Distance_meters.*sind(data.diff));
                
                % Check if target is within contour
                in_footprint = inpolygon(target_x, target_y, contour_x, contour_y);
                
                if in_footprint
                    % Find footprint value at target point
                    [~, x_idx] = min(abs(x_vals - target_x));
                    [~, y_idx] = min(abs(y_vals - target_y));
                    footprint_value = F_normalized(y_idx, x_idx);
                else
                    footprint_value = NaN;
                end
                
                % Add results to table
                data.InFootprint = repmat(in_footprint, height(data), 1);
                data.FootprintValue2 = repmat(footprint_value, height(data), 1);
                
                % Save updated table
               if in_footprint
    writetable(data, fullfile(outputPath, files(k).name));
                end


                
catch ME
    warning('Error in file: %s\nMessage: %s', files(k).name, ME.message);
    
    % Enhanced error diagnostics
    fprintf('Array sizes at error:\n');
    whos X Y f_x sigma_y F
    fprintf('Mean values:\n mu=%.2f, L=%.2f, flux_length=%.2f\n', ...
            mean_mu, mean(data.L), mean_flux_length);

            end
        end
    end
end
